import React, { useState } from "react";
import { api } from "../config/api";
import {
  Plus,
  Trash2,
  Save,
  GripVertical,
  Type,
  Hash,
  Mail,
  FileText,
  List,
} from "lucide-react";
import { useNavigate } from "react-router-dom";

export const Builder = () => {
  const [form, setForm] = useState({
    title: "",
    steps: [],
  });
  const [activeStep, setActiveStep] = useState(0);
  const navigate = useNavigate();

  const addStep = () => {
    const newStep = { title: "New Step", fields: [] };
    setForm({
      ...form,
      steps: [...form.steps, newStep],
    });
    setActiveStep(form.steps.length);
  };

  const addField = (stepIndex, type) => {
    const updatedSteps = [...form.steps];
    updatedSteps[stepIndex].fields.push({
      label: "New Question",
      type: type,
      options: type === "select" ? ["Option 1", "Option 2"] : [],
    });
    setForm({ ...form, steps: updatedSteps });
  };

  const updateStepTitle = (index, value) => {
    const updatedSteps = [...form.steps];
    updatedSteps[index].title = value;
    setForm({ ...form, steps: updatedSteps });
  };

  const updateField = (stepIndex, fieldIndex, key, value) => {
    const updatedSteps = [...form.steps];
    updatedSteps[stepIndex].fields[fieldIndex][key] = value;
    setForm({ ...form, steps: updatedSteps });
  };

  const updateFieldOptions = (stepIndex, fieldIndex, value) => {
    const updatedSteps = [...form.steps];
    updatedSteps[stepIndex].fields[fieldIndex].options = value
      .split(",")
      .map((opt) => opt.trim());
    setForm({ ...form, steps: updatedSteps });
  };

  const deleteStep = (index) => {
    const updatedSteps = form.steps.filter((_, i) => i !== index);
    setForm({ ...form, steps: updatedSteps });
    if (activeStep >= index && activeStep > 0) setActiveStep(activeStep - 1);
  };

  const deleteField = (stepIndex, fieldIndex) => {
    const updatedSteps = [...form.steps];
    updatedSteps[stepIndex].fields = updatedSteps[stepIndex].fields.filter(
      (_, i) => i !== fieldIndex,
    );
    setForm({ ...form, steps: updatedSteps });
  };

  const saveForm = async () => {
    try {
      const res = await api.post("/api/forms", form);
      navigate("/");
    } catch (err) {
      console.error(err);
      alert("Error saving form");
    }
  };

  const FieldTypeButton = ({ type, icon: Icon, label }) => (
    <button
      onClick={() => addField(activeStep, type)}
      className="flex items-center gap-2 w-full p-3 rounded hover:bg-neutral-100 transition text-left text-sm font-medium text-neutral-700"
    >
      <Icon size={16} />
      {label}
    </button>
  );

  return (
    <div className="flex h-screen bg-white overflow-hidden font-sans">
      {/* Sidebar - Steps */}
      <div className="w-64 bg-neutral-50 border-r border-neutral-200 flex flex-col">
        <div className="p-4 border-b border-neutral-200">
          <h1 className="font-bold text-lg text-neutral-800">Form Builder</h1>
        </div>
        <div className="flex-1 overflow-y-auto p-2 space-y-1">
          {form.steps.map((step, index) => (
            <div
              key={index}
              onClick={() => setActiveStep(index)}
              className={`flex items-center gap-2 p-3 rounded cursor-pointer group ${activeStep === index ? "bg-blue-100 text-blue-700" : "hover:bg-neutral-100 text-neutral-600"}`}
            >
              <div className="w-6 h-6 rounded-full bg-white border border-neutral-200 flex items-center justify-center text-xs font-bold shadow-sm">
                {index + 1}
              </div>
              <span className="truncate text-sm font-medium flex-1">
                {step.title}
              </span>
              <button
                onClick={(e) => {
                  e.stopPropagation();
                  deleteStep(index);
                }}
                className="opacity-0 group-hover:opacity-100 text-red-400 hover:text-red-600 p-1"
              >
                <Trash2 size={14} />
              </button>
            </div>
          ))}
          <button
            onClick={addStep}
            className="flex items-center gap-2 w-full p-3 rounded hover:bg-neutral-100 text-neutral-500 text-sm font-medium transition"
          >
            <Plus size={16} />
            Add Step
          </button>
        </div>
        <div className="p-4 border-t border-neutral-200">
          <button
            onClick={saveForm}
            className="flex items-center justify-center gap-2 w-full py-2 bg-neutral-900 text-white rounded font-medium hover:bg-neutral-800 transition"
          >
            <Save size={16} />
            Publish
          </button>
        </div>
      </div>

      {/* Main Content - Canvas */}
      <div className="flex-1 flex flex-col bg-neutral-100">
        <div className="bg-white border-b border-neutral-200 px-8 py-4 flex items-center justify-between">
          <input
            className="text-xl font-medium outline-none text-neutral-700 w-full bg-transparent"
            placeholder="Form Title"
            value={form.title}
            onChange={(e) => setForm({ ...form, title: e.target.value })}
          />
        </div>

        <div className="flex-1 overflow-y-auto p-12 flex justify-center">
          {form.steps[activeStep] ? (
            <div className="w-full h-full overflow-y-scroll max-w-2xl bg-white rounded-xl shadow-sm border border-neutral-200 p-8 animate-fadeIn">
              <div className="mb-8">
                <label className="block text-xs font-bold text-neutral-400 uppercase tracking-widest mb-2">
                  Step Title
                </label>
                <input
                  className="w-full text-3xl font-light outline-none border-b border-neutral-200 pb-2 focus:border-blue-500 transition"
                  value={form.steps[activeStep].title}
                  onChange={(e) => updateStepTitle(activeStep, e.target.value)}
                  placeholder="e.g. Personal Details"
                />
              </div>

              <div className="space-y-6">
                {form.steps[activeStep].fields.map((field, fIndex) => (
                  <div
                    key={fIndex}
                    className="bg-neutral-50 p-6 rounded-lg border border-neutral-200 relative group"
                  >
                    <button
                      onClick={() => deleteField(activeStep, fIndex)}
                      className="absolute top-2 right-2 text-neutral-400 hover:text-red-500 opacity-0 group-hover:opacity-100 transition"
                    >
                      <Trash2 size={16} />
                    </button>

                    <input
                      className="w-full bg-transparent font-medium text-lg outline-none mb-4 text-neutral-700 placeholder:text-neutral-400"
                      value={field.label}
                      placeholder="Question Text"
                      onChange={(e) =>
                        updateField(activeStep, fIndex, "label", e.target.value)
                      }
                    />

                    {field.type === "select" && (
                      <div className="mt-2">
                        <label className="block text-xs font-bold text-neutral-400 uppercase mb-1">
                          Options
                        </label>
                        <input
                          className="w-full border border-neutral-300 rounded px-3 py-2 text-sm bg-white"
                          value={field.options.join(", ")}
                          onChange={(e) =>
                            updateFieldOptions(
                              activeStep,
                              fIndex,
                              e.target.value,
                            )
                          }
                          placeholder="Option 1, Option 2..."
                        />
                      </div>
                    )}

                    <div className="mt-4 flex gap-4 text-xs font-bold text-neutral-400 uppercase tracking-wider">
                      <div className="flex items-center gap-1">
                        {field.type === "text" && <Type size={12} />}
                        {field.type === "number" && <Hash size={12} />}
                        {field.type === "email" && <Mail size={12} />}
                        {field.type === "textarea" && <FileText size={12} />}
                        {field.type === "select" && <List size={12} />}
                        {field.type}
                      </div>
                    </div>
                  </div>
                ))}
              </div>

              <div className="mt-8 border-t border-neutral-200 pt-6">
                <label className="block text-xs font-bold text-neutral-400 uppercase tracking-widest mb-4">
                  Add Question
                </label>
                <div className="grid grid-cols-2 md:grid-cols-3 gap-2">
                  <FieldTypeButton type="text" icon={Type} label="Short Text" />
                  <FieldTypeButton
                    type="textarea"
                    icon={FileText}
                    label="Long Text"
                  />
                  <FieldTypeButton type="number" icon={Hash} label="Number" />
                  <FieldTypeButton type="email" icon={Mail} label="Email" />
                  <FieldTypeButton type="select" icon={List} label="Dropdown" />
                </div>
              </div>
            </div>
          ) : (
            <div className="flex flex-col items-center justify-center text-neutral-400">
              <p>Select or create a step to start editing</p>
            </div>
          )}
        </div>
      </div>
    </div>
  );
};
